/*  TRADE ANALYSIS PROGRAM
**  Carries out the Beck/Katz calculation of panel-corrected time series-cross section analysis
**  Begins with an OLS run on the data [including dummy variables for the dyads - really?].
**  Rho is estimated from the Durbin-Watson statistic.
**  Data is corrected with Prais-Winston, and then re-run with corrected standard errors.
**  Adapted from program provided by James Morrow and co-authors of 1998 paper.
**  Assumes matrix sorted by dyad, then year, with years same for each dyad, first two cols reporting dyad & year.
*/

/* USER MUST CHANGE THE ITEMS IMMEDIATELY BELOW */

/* Establishes name of data set */
inname = "cases7";

/* Opens file to capture text output */
output file = cases7.out reset;

/* Sets matrix size; assumes every dyad has same number of rows */
numyears=1997-1948+1;  Print "# of years in data set:  " numyears;
numctry=50; Print "# of countries in data set:  " numctry;
thecols=numctry*(numctry-1);  Print "# of dyads in data set:  " thecols;
maxrows=thecols*numyears;  Print "# of rows in data set:  " maxrows;

/* Establishes the various variable names */
dyadlab = { dyad };
yearlab = { year };
exports = { genexp };
exog = { gdpi2, gdpj2, popi2, popj2, dist2, langpct2, relgpct2, mid2, allyd2, alltau2, demd2, demi2, demj2 };

/* END OF ITEMS THAT USER MUST CHANGE */

/* OLS globals: Use constant, Compute residuals and Durbin-Watson, Listwise deletion */
__con = 1;
_olsres = 1;
__miss = 1;

print "Ready to run OLS";

/* OLS regression to determine serial correlation
NOTE: MORROW ET AL. RAN ON ENTIRE DATA SET WITH NO REGARD FOR CROSS-SECS!!! */
{ vnam,m,b,stb,vc,stderr,sigma,cx,rsq,resid,dwstat } = ols(inname,exports,exog);
rho = 1 - dwstat/2;
print "Rho = " rho;

/* Opens data set for subsequent use */
open f2 = ^inname for read;

/* Pulls in variable names, then grabs indices where the variables, by name, appear in matrix */
{ name,indx } = indices(inname,0);
dyadind = indcv(dyadlab,name);
yrind = indcv(yearlab,name);
xinds = indcv(exog,name);
yind = indcv(exports,name);

/* Pulls in entire data set, then closes file handle */
fullmat = readr(f2,maxrows);
close(f2);

/* Creates initial variable matrices; adds constant term to ind vars */
dyad = fullmat[.,dyadind];
year = fullmat[.,yrind];
y = fullmat[.,yind];
x = ones(rows(fullmat),1)~fullmat[.,xinds];

/* Creates n x 1 vector identifying when rows begin a new sequence */
begin = year[2:rows(year),1] ./= (year[1:(rows(year)-1),1] + 1);

/* C-O tranform; Prais-Winsten at beg of each, with special one for very first row */
/* Note: Dropped this stage because Horace said he already did the C-O/P-W on the data */
pwx = x;
pwy = y;

/* dumps the unneeded matrices */
clear fullmat; clear x; clear y;

/* Creates variable names for all parts of the equation */
labels = dyadind | yrind | yind | "constant" | exog;

/* Creates variable names for only explanatory vars of the equation */
goxe = "constant" | exog;

/* Reruns ols with no constant, using temptrn's P-W data, dep var id, list of ind vars */
__con = 0;
{ vnam,m,b,stb,vc,stderr,sigma,cx,rsq,resid,dwstat } = ols(0,pwy,pwx);

/* Computes residuals using results of new OLS regression */
    resid = pwy - pwx*b;

/* Changes residual vector so that missing data become zeroes */
resid = missrv(resid,0);

/* Turns the residuals into a matrix, 1 col per dyad */
resid = reshape(resid,numyears,thecols);

/* Counts the number of rows in each resid column with a non-zero value */
crncases = sumc(resid ./= 0);

/* Averages the non-zero residuals */
crres = (sumc(resid))./crncases;

/* Two-tailed test to see probability of getting a mean residual
of this size given the variation in that mean: absolute value
of residual divided by sigma/sqrt(n) with n degrees of freedom;
uses t distrib */
cresprob = 2*cdftc(abs(crres).*sqrt(crncases)/sigma,crncases);

print "Long Stage #1 completed";

/* Computes T analog from the Beck and Katz piece, but I do
not understand how this computation gives this */

smog = (crncases*ones(1,thecols))
.*
(crncases*ones(1,thecols).<=(crncases*ones(1,thecols))')
     +
((crncases*ones(1,thecols))')
.*
 (crncases*ones(1,thecols).>(crncases*ones(1,thecols))');

print "Long Stage #2 Completed";

/* Computes estimated Sigma for pcse */
sgma = (resid'resid)./smog;

print "Long State #3 completed";

/* Replaces with 0's all MD codes among the explanatory vars */
xxx = missrv(pwx,0);

/* worthless I think */  j = 1;

/* Omega starts as a k*1 vector of ones */
omega = ones(cols(xxx),1);

/* Sets up two counters, k for # of cols, j for # of rows */
k = 1;
do until k > thecols;
    j = 1;
    do until j > numyears;

/* creates a matrix of ones, then eliminates all values under
the diagonal, then sums the columns to create a col vector counting
from 1 to thecols, then takes ones away, then multiplies by
numyears (# of years in data) and adds the j counter value */
        crap = (numyears*(sumc(upmat(ones(thecols,thecols)))-1)) + j;

/* Is this an indirect way to produce a Kronecker product?
Why not use the regular GAUSS operator?  */

/* attaches to right of initial vector; uses all rows of xxx
listed in crap (so always 42) to add on k x k matrices
each time */
        omega = omega~(xxx[crap,.]'sgma[.,k]);
        j = j+1;
    endo;
    k = k+1;
endo;

omega = omega[.,2:cols(omega)];

xxxx = inv(moment(pwx,1));

/* Calculates panel-corrected standard errors */
pcse = sqrt(abs(diag(xxxx*omega*xxx*xxxx)));

/* Calculates new t stats based upon the conservative std errs */
t = b./pcse;

/* Calculates new p vals based upon the conservative std errs */
pvt = 2*cdftc(abs(t),rows(pwy)-rows(t));

print;
print "                         Standard                 Prob";
print "Variable     Estimate      Error      t-value     >|t|";


print "------------------------------------------------------------";

finres = goxe~b~pcse~t~pvt;

let mask[1,5] = 0 1 1 1 1;

let fmt[5,3] = "-*.*s" 9 8 "*.*lf" 12 6 "*.*lf" 12 6 "*.*lf" 12 6 ""\
            "*.*lf" 10 3;

yyy = printfm (finres,mask,fmt);
